/**
 * \file: exchnd_backend.c
 *
 * Back-ends common code.
 * The back-ends are expected to store messages, the preparation of these
 * messages as their destruction is common to all of them.
 *
 * \component: exchndd
 *
 * \author: Frederic Berat (fberat@de.adit-jv.com)
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#include <errno.h>
#include <malloc.h>
#include <pthread.h>
#include <stdarg.h>

#include <sys/prctl.h>

#include "exchnd_backend.h"
#include "exchndd.h"

static char *backend_file_name;

int backend_mask = DEFAULT_BACKEND;

pthread_cond_t new_msg = PTHREAD_COND_INITIALIZER;
pthread_cond_t msg_handled = PTHREAD_COND_INITIALIZER;

struct msg_list *mlist;
static struct msg_list mlist_head;
static ExchndBackend_t *backend;
static pthread_t exchnd_garbage_collector_th;
static int gc_skip_waiting;

/* \func exchnd_garbage_collector
 *
 * Garbage collection
 *
 * Destroy read messages and clean-up queue on exit.
 *
 * The garbage collection can't be done at the end of each thread.
 * If we would have done so, the slowest back-end would have been even more
 * slow.
 */
static void *exchnd_garbage_collector(void *arg)
{
    struct msg_list *current = mlist;
    pthread_mutex_t *msg_lck = &mlist->msg_lck;
    struct msg_list *next = NULL;

    (void)arg;

    pthread_mutex_lock(msg_lck);
    prctl(PR_SET_NAME, "garbage_collector", NULL, NULL, NULL);

    while (!(action & EXCHND_TERMINATE)) {
        pthread_cond_wait(&new_msg, msg_lck);

        current = mlist->next;

        while ((current != mlist) &&
               ((current->read_flag & ALL_BACKEND) == ALL_BACKEND)) {
            next = current->next;
            exchnd_list_del(current);

            pthread_mutex_unlock(msg_lck);

            free(current->msg);
            current->msg = NULL;
            free(current);
            current = next;

            pthread_mutex_lock(msg_lck);
        }
    }

    if (!gc_skip_waiting)
        pthread_cond_wait(&msg_handled, msg_lck);

    while (current != mlist) {
        next = current->next;
        exchnd_list_del(current);

        free(current->msg);
        free(current);
        current = next;
    }

    pthread_mutex_unlock(msg_lck);

    return NULL;
}

/* \func exchnd_wake_backends
 *
 * Wake-up back-ends
 *
 * Wake-up is expected to notify the end of processing:
 * "All messages are created, please proceed"
 *
 * The idea is to wake-up threads that may not have seen last messages.
 * That's most likely only useful for the garbage collector.
 */
void exchnd_wake_backends(void)
{
    pthread_mutex_lock(&mlist->msg_lck);

    /* Now broadcasting info to all backend's threads. */
    pthread_cond_broadcast(&new_msg);

    pthread_mutex_unlock(&mlist->msg_lck);
}

/* \func exchnd_wait_default_backend
 *
 * Wait for default back-end to read messages
 *
 * If the driver plans to restart the board, we need to ensure that all
 * messages are at least stored in the default back-end.
 * Then, notify for "fake" new message and wait for it to answer.
 */
void exchnd_wait_default_backend(void)
{
    pthread_mutex_lock(&mlist->msg_lck);

    /* Now broadcasting info to all backend's threads. */
    pthread_cond_broadcast(&new_msg);

    /* Wait until default backend has finished */
    pthread_cond_wait(&msg_handled, &mlist->msg_lck);

    pthread_mutex_unlock(&mlist->msg_lck);
}

static char *exchnd_append_header(char *temp, char *in)
{
    unsigned int size = strlen(in);
    char *org = temp;

    while (size--) {
        if ((temp == org) || (*(temp - 1) == '\n')) {
            *temp++ = 'E';
            *temp++ = 'X';
            *temp++ = 'H';
            *temp++ = ' ';
        }

        *temp++ = *in++;
    }

    if (*(temp - 1) != '\n')
        *temp = '\n';

    return org;
}

#define EXH(x, y) exchnd_append_header(x, y)

/*
 * \func exchnd_backend_store
 *
 * Write a message to all configured backends
 *
 * \param msg Data to be written in the back-ends
 * \param len Length of the data
 */
void exchnd_backend_store(char *msg, int len)
{
    struct msg_list *next_mlist = NULL;
    char *local;

    if (len <= 1)
        return;

    next_mlist = malloc(sizeof(struct msg_list));

    if (!next_mlist) {
        exchnd_print_error("No memory available for %s", __func__);
        return;
    }

    memset(next_mlist, 0, sizeof(struct msg_list));

    /* Allocate 3 times size in order to be able to put
     * the "EXH " header on each line.
     */
    local = calloc(3, len + 1);

    if (!local) {
        exchnd_print_error("No memory available for %s", __func__);
        free(next_mlist);
        return;
    }

    EXH(local, msg);

    pthread_mutex_init(&next_mlist->msg_lck, NULL);

    /* Adding back null character. */
    next_mlist->msg = local;
    next_mlist->len = strlen(local);

    /* Consider msg as read for not created backend; */
    next_mlist->read_flag = (ALL_BACKEND & ~backend_mask);

    pthread_mutex_lock(&mlist->msg_lck);
    /* Put the message in queue. */
    exchnd_list_add_tail(next_mlist, mlist);

    /* Now broadcasting info to all backend's threads. */
    pthread_cond_broadcast(&new_msg);

    pthread_mutex_unlock(&mlist->msg_lck);

    return;
}

/*
 * \func exchnd_log_error
 *
 * This function writes an error message into the. It provides a
 * printf interface to add parameters.
 *
 * \param format Format string for the message to write
 * \param ... Further parameters
 */
/* PRQA: Lint Message 530: False positive, the va_start is initializing args */
/*lint -save -e530 */
void exchnd_log_error(const char *format, ...)
{
    char buffer[MAX_MSG_LEN];
    va_list args;

    va_start(args, format);
    vsnprintf(buffer, MAX_MSG_LEN, format, args);
    exchnd_backend_store(buffer, strlen(buffer));
    va_end(args);
}
/*lint -restore */

void exchnd_set_file_name(char *name)
{
    backend_file_name = calloc(1, 4096);

    strncpy(backend_file_name, name, 4096);
}

int exchnd_init_storage(void)
{
    int ret = 0;

    mlist = &mlist_head;
    gc_skip_waiting = 0;

    pthread_mutex_init(&mlist->msg_lck, NULL);
    exchnd_init_list(mlist);

    ret = exchnd_create_thread(&exchnd_garbage_collector_th,
                               exchnd_garbage_collector,
                               NULL,
                               EXH_NORMAL_TH);

    if (ret) {
        exchnd_print_error("Unable to initialize storage: %s", strerror(errno));
        return ret;
    }

    /* Taking lock so that we can ensure Default backend is ready */
    pthread_mutex_lock(&mlist->msg_lck);

    if (backend_mask & FILE_BACKEND)
        /* log messages to the console */
        backend = exchnd_backend_create_file(backend, backend_file_name);

#ifdef DLT

    if (backend_mask & DLT_BACKEND)
        /* log messages to DLT */
        backend = exchnd_backend_create_dlt(backend);

#endif /* DLT */

    if (backend_mask & ERRMEM_BACKEND)
        backend = exchnd_backend_create_errmem(backend);

    /* Now waiting for default backend to be ready */
    pthread_cond_wait(&msg_handled, &mlist->msg_lck);
    pthread_mutex_unlock(&mlist->msg_lck);

    return ret;
}

void exchnd_cleanup_storage(void)
{
    /* Wake the back-ends and destroy them. */
    exchnd_wake_backends();

    /* clean up backends */
    while (backend != NULL) {
        ExchndBackend_t *old_backend = backend;
        backend = backend->next;
        old_backend->destroy(old_backend);
    }

    /* Now we can clean last messages. */
    pthread_mutex_lock(&mlist->msg_lck);
    gc_skip_waiting = 1;
    pthread_cond_broadcast(&msg_handled);
    pthread_mutex_unlock(&mlist->msg_lck);
    pthread_join(exchnd_garbage_collector_th, NULL);

    free(backend_file_name);
}